﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/

using System;
using System.Collections.Generic;
//using System.Linq;
//using System.Text;
using System.Xml;
using Microsoft.Xna.Framework.Input;

namespace GE.Input
{
    /// <summary>
    /// Represent the keyboard.
    /// </summary>
    class KeyBoard : Device
    {
        /// <summary>
        /// Maps between an input mode and the actions and then between inputs and game actions
        /// </summary
        private Dictionary<InputMode, Dictionary<Keys, GameAction>> _actions;

        /// <summary>
        /// State of the keyboard
        /// </summary>
        private KeyboardState _keyboardState;

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="device">xml node of the device</param>
        public KeyBoard(XmlNode device)
        {
            _actions = new Dictionary<InputMode, Dictionary<Keys, GameAction>>();
            _keyboardState = Keyboard.GetState(0);

            loadDevice(device);
        }

        /// <summary>
        /// Load the mapping between inputs and actions
        /// </summary>
        /// <param name="device"></param>
        public override void loadDevice(XmlNode device)
        {
            //go throught avery game mode
            XmlNodeList modes = device.ChildNodes;
            foreach (XmlNode mode in modes)
            {
                if (mode.Name.CompareTo("mode") == 0)
                {
                    //create the action dictionnary for the new mode
                    InputMode imode = (InputMode)Enum.Parse(typeof(InputMode), mode.Attributes["name"].Value, true);
                    _actions[imode] = new Dictionary<Keys, GameAction>();

                    //go throught every action
                    XmlNodeList actions = mode.ChildNodes;
                    foreach (XmlNode action in actions)
                    {
                        if (action.Name.CompareTo("action") != 0) continue;
                        
                        //create the new action
                        Keys iaction = (Keys)Enum.Parse(typeof(Keys), action.Attributes["inputAction"].Value, true);
                        GameAction gaction = (GameAction)Enum.Parse(typeof(GameAction),
                            action.Attributes["gameAction"].Value, true);

                        _actions[imode][iaction] = gaction;
                        
                    }
                }
               
            }
        }

        /// <summary>
        /// Update the keyboard state and send messages
        /// </summary>
        public override void update(bool bSendRealease)
        {
            if(bSendRealease)
                updateWithRelease();
            else
                updateWithoutRelease();
        }

        private void updateWithoutRelease()
        {
            //update the keyboard state
            _keyboardState = Keyboard.GetState(0);

            //get the dictionnary containing the actions for the game mode
            Dictionary<Keys, GameAction> map = _actions[Input.Instance.Mode];

            //go throught every action of the game
            foreach (Keys k in map.Keys)
            {
                //check if the key is pressed
                if (_keyboardState.IsKeyDown(k))
                    Input.Instance.sendInputAction(map[k], 0, 1, 0);
            }
        }

        private void updateWithRelease()
        {
            //update the keyboard state
            _keyboardState = Keyboard.GetState(0);

            //get the dictionnary containing the actions for the game mode
            Dictionary<Keys, GameAction> map = _actions[Input.Instance.Mode];

            //go throught every action of the game
            foreach (Keys k in map.Keys)
            {
                //check if the key is pressed
                if (!_keyboardState.IsKeyDown(k))
                    Input.Instance.sendInputAction(map[k], 0, 0, 0);
                else
                    Input.Instance.sendInputAction(map[k], 0, 1, 0);
            }
        }
    }
}
